#!/usr/bin/env python3
"""
SQreamDB MCP Server Setup Script

This script automates the setup process for the SQreamDB MCP server:
1. Creates a virtual environment
2. Installs required dependencies
3. Creates Claude desktop configuration

Usage: python setup_sqreamdb_mcp.py host=192.168.4.68 port=5000 database=master username=sqream password=sqream
"""

import sys
import os
import subprocess
import json
import venv
import shutil
from pathlib import Path
import platform

class SQreamDBMCPSetup:
    def __init__(self, connection_params):
        self.connection_params = connection_params
        self.script_dir = Path(__file__).parent.absolute()
        self.venv_dir = self.script_dir / "venv"
        self.server_script = self.script_dir / "sqreamdb_mcp_server.py"
        
        # Check if server script exists
        if not self.server_script.exists():
            raise FileNotFoundError(f"Server script not found: {self.server_script}")
        
        # Determine Claude config directory based on OS
        self.claude_config_dir = self._get_claude_config_dir()
        
    def _get_claude_config_dir(self):
        """Get Claude desktop config directory based on operating system."""
        system = platform.system().lower()
        
        if system == "windows":
            # Windows: %APPDATA%\Claude
            return Path(os.environ.get('APPDATA', '')) / "Claude"
        elif system == "darwin":  # macOS
            # macOS: ~/Library/Application Support/Claude
            return Path.home() / "Library" / "Application Support" / "Claude"
        else:  # Linux and others
            # Linux: ~/.config/claude
            return Path.home() / ".config" / "Claude"
    
    def create_virtual_environment(self):
        """Create a virtual environment for the MCP server."""
        print(f"Creating virtual environment at {self.venv_dir}...")
        
        if self.venv_dir.exists():
            print("Virtual environment already exists. Removing and recreating...")
            shutil.rmtree(self.venv_dir)
        
        # Create the virtual environment
        venv.create(self.venv_dir, with_pip=True)
        print("✓ Virtual environment created successfully")
    
    def get_python_executable(self):
        """Get the path to the Python executable in the virtual environment."""
        if platform.system().lower() == "windows":
            return self.venv_dir / "Scripts" / "python.exe"
        else:
            return self.venv_dir / "bin" / "python"
    
    def get_pip_executable(self):
        """Get the path to the pip executable in the virtual environment."""
        if platform.system().lower() == "windows":
            return self.venv_dir / "Scripts" / "pip.exe"
        else:
            return self.venv_dir / "bin" / "pip"
    
    def install_dependencies(self):
        """Install required Python packages in the virtual environment."""
        print("Installing dependencies...")
        
        pip_path = self.get_pip_executable()
        
        # List of required packages
        packages = [
            "pysqream",
            "aiohttp",
            "mcp"
        ]
        
        for package in packages:
            print(f"Installing {package}...")
            result = subprocess.run([
                str(pip_path), "install", package
            ], capture_output=True, text=True)
            
            if result.returncode != 0:
                print(f"Error installing {package}:")
                print(result.stderr)
                return False
            else:
                print(f"✓ {package} installed successfully")
        
        return True
    
    def create_claude_config(self):
        """Create or update Claude desktop configuration."""
        print("Creating Claude desktop configuration...")
        
        # Ensure Claude config directory exists
        self.claude_config_dir.mkdir(parents=True, exist_ok=True)
        
        config_file = self.claude_config_dir / "claude_desktop_config.json"
        
        # Get the Python executable from our venv
        python_executable = str(self.get_python_executable())
        server_script_path = str(self.server_script)
        connection_string = " ".join([f"{k}={v}" for k, v in self.connection_params.items()])
        
        # Create the configuration
        config = {
            "mcpServers": {
                "sqreamdb": {
                    "command": python_executable,
                    "args": [
                        server_script_path,
                        connection_string
                    ]
                }
            }
        }
        
        # If config file already exists, merge with existing configuration
        if config_file.exists():
            print("Existing Claude configuration found. Merging...")
            try:
                with open(config_file, 'r', encoding='utf-8') as f:
                    existing_config = json.load(f)
                
                # Merge mcpServers section
                if "mcpServers" not in existing_config:
                    existing_config["mcpServers"] = {}
                
                existing_config["mcpServers"]["sqreamdb"] = config["mcpServers"]["sqreamdb"]
                config = existing_config
                
            except (json.JSONDecodeError, KeyError) as e:
                print(f"Warning: Could not parse existing config file. Creating new one. Error: {e}")
        
        # Write the configuration
        with open(config_file, 'w', encoding='utf-8') as f:
            json.dump(config, f, indent=2)
        
        print(f"✓ Claude configuration created/updated at {config_file}")
        
        # Print the configuration for reference
        print("\n" + "="*60)
        print("CLAUDE DESKTOP CONFIGURATION:")
        print("="*60)
        print(json.dumps(config, indent=2))
        print("="*60)
        
        # Add helpful troubleshooting message
        print("\n📋 TROUBLESHOOTING:")
        print("If you don't see your SQreamDB connection working after restarting Claude Desktop,")
        print("please navigate to Claude Desktop → Settings → Developer and manually edit")
        print("the configuration with the JSON shown above.")
        print(f"\nConfiguration file location: {config_file}")
        print("\n⚠️  Remember to restart Claude Desktop completely after making changes!")
        
        return config_file
    
    def test_connection(self):
        """Test the database connection."""
        print("Testing database connection...")
        
        python_executable = self.get_python_executable()
        
        test_script = f'''
import sys
try:
    import pysqream
    conn_params = {self.connection_params}
    
    print("Attempting to connect to SQreamDB...")
    connection = pysqream.connect(**conn_params)
    
    with connection.cursor() as cursor:
        cursor.execute("SELECT 1 as test_column")
        result = cursor.fetchone()
        print(f"✓ Connection successful! Test query result: {{result}}")
    
    connection.close()
    print("Connection test passed!")
except Exception as e:
    print(f"✗ Connection failed: {{e}}")
    sys.exit(1)
'''
        
        result = subprocess.run([
            str(python_executable), "-c", test_script
        ], capture_output=True, text=True)
        
        print(result.stdout)
        if result.stderr:
            print("Errors:", result.stderr)
        
        return result.returncode == 0
    
    def run_setup(self):
        """Run the complete setup process."""
        print("=" * 60)
        print("SQreamDB MCP Server Setup")
        print("=" * 60)
        print(f"Connection parameters: {self.connection_params}")
        print(f"Setup directory: {self.script_dir}")
        print(f"Server script: {self.server_script}")
        print(f"Claude config directory: {self.claude_config_dir}")
        print()
        
        try:
            # Step 1: Create virtual environment
            self.create_virtual_environment()
            
            # Step 2: Install dependencies
            if not self.install_dependencies():
                print("✗ Failed to install dependencies")
                return False
        
            
            # Step 4: Create Claude configuration
            config_file = self.create_claude_config()
            
            print()
            print("=" * 60)
            print("✓ Setup completed successfully!")
            print("=" * 60)
            print(f"Virtual environment: {self.venv_dir}")
            print(f"Server script: {self.server_script}")
            print(f"Claude config: {config_file}")
            print()
            print("Next steps:")
            print("1. Restart Claude Desktop application")
            print("2. The SQreamDB MCP server should now be available in Claude")
            print("3. You can test it by asking Claude to execute SQreamDB queries")
            print()
            
            return True
            
        except Exception as e:
            print(f"✗ Setup failed with error: {e}")
            import traceback
            traceback.print_exc()
            return False


def parse_connection_string(connection_string):
    """Parse connection string into parameters dictionary."""
    params = {}
    
    for part in connection_string.split():
        if '=' not in part:
            continue
        key, value = part.split('=', 1)
        
        # Convert port to integer
        if key.lower() == 'port':
            try:
                params[key] = int(value)
            except ValueError:
                raise ValueError(f"Port must be a number, got: {value}")
        else:
            params[key] = value
    
    # Validate required parameters
    required_params = ['host', 'port', 'database', 'username', 'password']
    missing_params = [param for param in required_params if param not in params]
    
    if missing_params:
        raise ValueError(f"Missing required parameters: {', '.join(missing_params)}")
    
    return params


def print_help():
    """Print detailed help information."""
    help_text = """
╔══════════════════════════════════════════════════════════════════════════════╗
║                        SQreamDB MCP Server Setup                            ║
╚══════════════════════════════════════════════════════════════════════════════╝

This script sets up the SQreamDB MCP (Model Context Protocol) server for use 
with Claude Desktop. It will:

  1. Create a Python virtual environment
  2. Install required dependencies (pysqream, aiohttp, mcp)
  3. Test the database connection
  4. Configure Claude Desktop to use the MCP server

USAGE:
  python setup_sqreamdb_mcp.py [OPTIONS] <connection_parameters>

CONNECTION PARAMETERS (all required):
  host=<hostname>       Database server hostname or IP address
  port=<port_number>    Database server port (typically 5000)
  database=<db_name>    Database name to connect to
  username=<username>   Database username
  password=<password>   Database password

OPTIONS:
  -h, --help           Show this help message and exit

EXAMPLES:
  # Basic setup
  python setup_sqreamdb_mcp.py host=192.168.4.68 port=5000 database=master username=sqream password=sqream

  # Local database
  python setup_sqreamdb_mcp.py host=localhost port=5000 database=test_db username=admin password=mypass

  # Remote database with custom port
  python setup_sqreamdb_mcp.py host=sqream.mycompany.com port=3108 database=analytics username=analyst password=secret123

WHAT HAPPENS AFTER SETUP:
  1. Restart Claude Desktop application
  2. Ask Claude to execute SQreamDB queries - it will automatically use the MCP server
  3. Example: "Show me all tables in my SQreamDB database"

TROUBLESHOOTING:
  • Ensure SQreamDB server is running and accessible
  • Check firewall settings if connecting to remote database
  • Verify credentials are correct
  • Make sure sqreamdb_mcp_server.py is in the same folder as this setup script

REQUIREMENTS:
  • Python 3.7 or higher
  • Network access to SQreamDB server
  • sqreamdb_mcp_server.py file in the same directory

For more information about SQreamDB, visit: https://docs.sqream.com/
"""
    print(help_text)


def main():
    """Main entry point for setup script."""
    # Check for help flags
    if len(sys.argv) == 2 and sys.argv[1] in ['-h', '--help', 'help']:
        print_help()
        sys.exit(0)
    
    if len(sys.argv) < 2:
        print("X Error: Missing connection parameters")
        print("\nUSAGE:")
        print("  python setup_sqreamdb_mcp.py host=... port=... database=... username=... password=...")
        print("\nEXAMPLE:")
        print("  python setup_sqreamdb_mcp.py host=192.168.4.68 port=5000 database=master username=sqream password=sqream")
        print("\nFor detailed help, run:")
        print("  python setup_sqreamdb_mcp.py --help")
        sys.exit(1)
    
    try:
        # Join all arguments after the script name into one connection string
        connection_string = " ".join(sys.argv[1:])
        connection_params = parse_connection_string(connection_string)
        
        setup = SQreamDBMCPSetup(connection_params)
        success = setup.run_setup()
        
        sys.exit(0 if success else 1)
        
    except Exception as e:
        print(f"X Error: {e}")
        print("\nFor help, run:")
        print("  python setup_sqreamdb_mcp.py --help")
        sys.exit(1)


if __name__ == "__main__":
    main()
